﻿namespace Microsoft.Samples.PlanMyNight.Data.Tests
{
    using System;
    using System.Linq;
    using System.Transactions;
    using Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;

    [TestClass]
    public class ItinerariesRepositoryFixture
    {
        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldInsertNewItinerary()
        {
            var itinerary = new Itinerary() { Name = "My Test Itinerary", Created = DateTime.Now, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity()
            {
                ActivityId = "1",
                Order = 1,
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Latitude = 48.545691,
                Longitude = -117.891798,
                Zip = "99114"
            });

            using (new TransactionScope())
            {
                var repository = new ItinerariesRepository();
                repository.Add(itinerary);
            }

            Assert.AreNotEqual(0, itinerary.Id);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldRetrieveItinerary()
        {
            var expected = new Itinerary()
            {
                Name = "My Test Itinerary",
                Created = DateTime.Today,
                UserId = Guid.NewGuid(),
                IsPublic = true,
                Description = "Test Description"
            };

            expected.Activities.Add(new ItineraryActivity()
            {
                ActivityId = "1",
                Order = 1,
                EstimatedMinutes = 30,
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Latitude = 48.545691,
                Longitude = -117.891798,
                Zip = "99114"
            });

            Itinerary actual;
            using (new TransactionScope())
            {
                var repository = new ItinerariesRepository();
                repository.Add(expected);

                actual = repository.Retrieve(expected.Id);
            }

            Assert.AreEqual(expected.Name, actual.Name);
            Assert.AreEqual(expected.Created, actual.Created);
            Assert.AreEqual(expected.Description, actual.Description);
            Assert.AreEqual(expected.IsPublic, actual.IsPublic);
            Assert.AreEqual(expected.UserId, actual.UserId);
            Assert.AreEqual(1, actual.Activities.Count);
            Assert.AreEqual(expected.Activities[0].ActivityId, actual.Activities[0].ActivityId);
            Assert.AreEqual(expected.Activities[0].Order, actual.Activities[0].Order);
            Assert.AreEqual(expected.Activities[0].EstimatedMinutes, actual.Activities[0].EstimatedMinutes);
            Assert.AreEqual(expected.Activities[0].Latitude, actual.Activities[0].Latitude);
            Assert.AreEqual(expected.Activities[0].Longitude, actual.Activities[0].Longitude);
            Assert.AreEqual(expected.Activities[0].Zip, actual.Activities[0].Zip);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldRetrieveUserItineraries()
        {
            using (new TransactionScope())
            {
                var expectedUserId = Guid.NewGuid();

                var expected1 = new Itinerary()
                {
                    Name = "My Test Itinerary #1",
                    Created = DateTime.Today,
                    UserId = expectedUserId,
                    IsPublic = true,
                    Description = "Test Description"
                };

                expected1.Activities.Add(new ItineraryActivity()
                {
                    ActivityId = "1",
                    Order = 1,
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });

                var expected2 = new Itinerary()
                {
                    Name = "My Test Itinerary #2",
                    Created = DateTime.Today,
                    UserId = expectedUserId,
                    IsPublic = true,
                    Description = "Test Description"
                };

                var other = new Itinerary()
                {
                    Name = "My Test Itinerary",
                    Created = DateTime.Today,
                    UserId = Guid.NewGuid(),
                    IsPublic = true,
                    Description = "Test Description"
                };

                var repository = new ItinerariesRepository();
                repository.Add(expected1);
                repository.Add(expected2);
                repository.Add(other);

                var result = repository.RetrieveByUser(expectedUserId);

                Assert.AreEqual(2, result.Count());
            }
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldUpdateItinerary()
        {
            var itinerary = new Itinerary()
            {
                Name = "My Original Test Itinerary",
                Created = DateTime.Today,
                UserId = Guid.NewGuid(),
                IsPublic = false,
                Description = "Original Test Description"
            };

            itinerary.Activities.Add(new ItineraryActivity()
            {
                ActivityId = "1",
                Order = 1,
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Latitude = 48.545691,
                Longitude = -117.891798,
                Zip = "99114"
            });

            Itinerary actual;
            using (new TransactionScope())
            {
                var repository = new ItinerariesRepository();
                repository.Add(itinerary);

                itinerary.Name = "New Name";
                itinerary.Description = "New Description";
                itinerary.IsPublic = true;
                itinerary.Activities[0].EstimatedMinutes = 30;
                itinerary.Activities.Add(new ItineraryActivity()
                {
                    ActivityId = "2",
                    Order = 2,
                    EstimatedMinutes = 60,
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });

                repository.Update(itinerary);

                actual = repository.Retrieve(itinerary.Id);
            }

            Assert.AreEqual(itinerary.Name, actual.Name);
            Assert.AreEqual(itinerary.Created, actual.Created);
            Assert.AreEqual(itinerary.Description, actual.Description);
            Assert.AreEqual(itinerary.IsPublic, actual.IsPublic);
            Assert.AreEqual(itinerary.UserId, actual.UserId);
            Assert.AreEqual(2, actual.Activities.Count);
            Assert.AreEqual(itinerary.Activities[0].ActivityId, actual.Activities[0].ActivityId);
            Assert.AreEqual(itinerary.Activities[0].Order, actual.Activities[0].Order);
            Assert.AreEqual(itinerary.Activities[0].EstimatedMinutes, actual.Activities[0].EstimatedMinutes);
            Assert.AreEqual(itinerary.Activities[0].Latitude, actual.Activities[0].Latitude);
            Assert.AreEqual(itinerary.Activities[0].Longitude, actual.Activities[0].Longitude);
            Assert.AreEqual(itinerary.Activities[0].Zip, actual.Activities[0].Zip);
            Assert.AreEqual(itinerary.Activities[1].ActivityId, actual.Activities[1].ActivityId);
            Assert.AreEqual(itinerary.Activities[1].Order, actual.Activities[1].Order);
            Assert.AreEqual(itinerary.Activities[1].EstimatedMinutes, actual.Activities[1].EstimatedMinutes);
            Assert.AreEqual(itinerary.Activities[1].Latitude, actual.Activities[1].Latitude);
            Assert.AreEqual(itinerary.Activities[1].Longitude, actual.Activities[1].Longitude);
            Assert.AreEqual(itinerary.Activities[1].Zip, actual.Activities[1].Zip);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldSearchItinerariesUsingActivity()
        {
            using (new TransactionScope())
            {
                var data = new PlanMyNightEntities();
                foreach (var itinerary in data.Itineraries)
                {
                    data.DeleteObject(itinerary);
                }

                data.SaveChanges();

                var expected1 = new Itinerary { Name = "Match #1", IsPublic = true, Created = DateTime.Now };
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "2",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected1);
                data.SaveChanges();

                var expected2 = new Itinerary { Name = "Match #2", IsPublic = true, Created = DateTime.Now };
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected2);
                data.SaveChanges();

                var other = new Itinerary { Name = "Other", IsPublic = true, Created = DateTime.Now };
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "4",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(other);
                data.SaveChanges();

                var repo = new ItinerariesRepository();
                var result = repo.SearchByActivity("1", 10, 1);

                Assert.AreEqual(2, result.TotalItems);
                Assert.AreEqual(1, result.CurrentPage);
                Assert.IsTrue(result.Items.Any(a => a.Name == "Match #1"));
                Assert.IsTrue(result.Items.Any(a => a.Name == "Match #2"));
            }
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldSearchByCity()
        {
            var repo = new ItinerariesRepository();
            var result = repo.SearchByCity(11199, "WA", "Colville", 10, 1);

            Assert.IsNotNull(result);
            Assert.AreEqual(1, result.TotalItems);
            Assert.AreEqual(1, result.TotalPages);
            Assert.AreEqual(10, result.PageSize);
            Assert.AreEqual(1, result.CurrentPage);
            Assert.IsNotNull(result.Items);
            Assert.AreEqual(1, result.Items.Count);
            Assert.IsTrue(result.Items.Any(i => 
                i.Name == "Colville Night Out" &&
                i.Activities.Count == 2));
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldSearchItinerariesUsingActivityAndReturnSecondPage()
        {
            using (new TransactionScope())
            {
                var data = new PlanMyNightEntities();
                foreach (var itinerary in data.Itineraries)
                {
                    data.DeleteObject(itinerary);
                }

                data.SaveChanges();

                var expected1 = new Itinerary { Name = "Match #1", IsPublic = true, Created = DateTime.Now };
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected1.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "2",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected1);
                data.SaveChanges();

                var expected2 = new Itinerary { Name = "Match #2", IsPublic = true, Created = DateTime.Now };
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                expected2.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "1",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(expected2);
                data.SaveChanges();

                var other = new Itinerary { Name = "Other", IsPublic = true, Created = DateTime.Now };
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "3",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                other.Activities.Add(new ItineraryActivity
                {
                    ActivityId = "4",
                    TypeId = 1,
                    State = "WA",
                    City = "Colville",
                    Latitude = 48.545691,
                    Longitude = -117.891798,
                    Zip = "99114"
                });
                data.Itineraries.AddObject(other);
                data.SaveChanges();

                var repo = new ItinerariesRepository();
                var result = repo.SearchByActivity("1", 1, 2);

                Assert.AreEqual(2, result.TotalItems);
                Assert.AreEqual(2, result.CurrentPage);
                Assert.AreEqual(1, result.Items.Count);
            }
        }

        // TODO: Update tests
        ////[TestMethod]
        ////[DeploymentItem("PlanMyNight.mdf")]
        ////public void ShouldSearchItinerariesUsingCriteria()
        ////{
        ////    Assert.Inconclusive();

        ////    ////Activity existingActivity = new ActivitiesRepository().Retrieve(1);

        ////    ////var expected = new Itinerary() { Name = "Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////expected.Activities.Add(new ItineraryActivity() { ActivityId = existingActivity.Id });

        ////    ////var searchCriteria = new AdvancedSearchQuery()
        ////    ////{
        ////    ////    ActivityTypeId = existingActivity.ActivityTypeId,
        ////    ////    City = existingActivity.City,
        ////    ////    Zip = existingActivity.Zip,
        ////    ////    StreetAddress = existingActivity.Street
        ////    ////};

        ////    ////PagingResult<Itinerary> actual;
        ////    ////using (new TransactionScope())
        ////    ////{
        ////    ////    var repository = new ItinerariesRepository();
        ////    ////    repository.Add(expected);

        ////    ////    actual = repository.Search(searchCriteria);
        ////    ////}

        ////    ////Assert.IsNotNull(actual);
        ////    ////Assert.AreNotEqual(0, actual.Items.Count);
        ////    ////Assert.IsTrue(actual.Items.Any(i => i.Id == expected.Id));
        ////}

        ////[TestMethod]
        ////[DeploymentItem("PlanMyNight.mdf")]
        ////public void ShouldNotGetItineraryThatDoesNotIncludeMatchingActivity()
        ////{
        ////    Assert.Inconclusive();

        ////    ////var notExpected = new Itinerary() { Name = "Not Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////notExpected.Activities.Add(new ItineraryActivity() { ActivityId = "1" });

        ////    ////var searchCriteria = new AdvancedSearchQuery()
        ////    ////{
        ////    ////    City = "NO-CITY"
        ////    ////};

        ////    ////PagingResult<Itinerary> actual;
        ////    ////using (new TransactionScope())
        ////    ////{
        ////    ////    var repository = new ItinerariesRepository();
        ////    ////    repository.Add(notExpected);

        ////    ////    actual = repository.Search(searchCriteria);
        ////    ////}

        ////    ////Assert.IsNotNull(actual);
        ////    ////Assert.AreEqual(0, actual.Items.Count);
        ////}

        ////[TestMethod]
        ////[DeploymentItem("PlanMyNight.mdf")]
        ////public void ShouldGetOnlyItinerariesForActivity()
        ////{
        ////    Assert.Inconclusive();

        ////    ////var expected = new Itinerary() { Name = "Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////expected.Activities.Add(new ItineraryActivity() { ActivityId = "1" });

        ////    ////var notExpected = new Itinerary() { Name = "Not Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////notExpected.Activities.Add(new ItineraryActivity() { ActivityId = "2" });

        ////    ////var searchCriteria = new AdvancedSearchQuery()
        ////    ////{
        ////    ////    ActivityTypeId = "1",
        ////    ////    City = string.Empty,
        ////    ////    Zip = string.Empty,
        ////    ////    StreetAddress = string.Empty
        ////    ////};

        ////    ////PagingResult<Itinerary> actual;
        ////    ////using (new TransactionScope())
        ////    ////{
        ////    ////    var repository = new ItinerariesRepository();
        ////    ////    repository.Add(expected);
        ////    ////    repository.Add(notExpected);

        ////    ////    actual = repository.Search(searchCriteria);
        ////    ////}

        ////    ////Assert.IsNotNull(actual);
        ////    ////Assert.AreNotEqual(0, actual.Items.Count);
        ////    ////Assert.IsTrue(actual.Items.Any(i => i.Id == expected.Id));
        ////    ////Assert.IsFalse(actual.Items.Any(i => i.Id == notExpected.Id));
        ////}

        ////[TestMethod]
        ////[DeploymentItem("PlanMyNight.mdf")]
        ////public void ShouldGetDistinctItinerariesForActivity()
        ////{
        ////    Assert.Inconclusive();

        ////    ////var searchCriteria = new AdvancedSearchQuery()
        ////    ////{
        ////    ////    City = "Westerville" // from mock data
        ////    ////};

        ////    ////var existingActivities = new ActivitiesRepository().Search(searchCriteria).Items;

        ////    ////var expected = new Itinerary() { Name = "Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////expected.Activities.Add(new ItineraryActivity() { ActivityId = existingActivities.ElementAt(0).Id });
        ////    ////expected.Activities.Add(new ItineraryActivity() { ActivityId = existingActivities.ElementAt(1).Id });

        ////    ////PagingResult<Itinerary> actual;
        ////    ////using (new TransactionScope())
        ////    ////{
        ////    ////    var repository = new ItinerariesRepository();
        ////    ////    repository.Add(expected);

        ////    ////    actual = repository.Search(searchCriteria);
        ////    ////}

        ////    ////Assert.IsNotNull(actual);
        ////    ////Assert.AreNotEqual(0, actual.Items.Count);
        ////    ////Assert.AreEqual(1, actual.Items.Count(i => i.Id == expected.Id));
        ////}

        ////[TestMethod]
        ////[DeploymentItem("PlanMyNight.mdf")]
        ////public void SearchShouldNotRetrieveActivitiesForItinerary()
        ////{
        ////    Assert.Inconclusive();

        ////    ////var expected = new Itinerary() { Name = "Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////expected.Activities.Add(new ItineraryActivity() { ActivityId = "1" });

        ////    ////var searchCriteria = new AdvancedSearchQuery()
        ////    ////{
        ////    ////    ActivityTypeId = "1",
        ////    ////    City = string.Empty,
        ////    ////    Zip = string.Empty,
        ////    ////    StreetAddress = string.Empty
        ////    ////};

        ////    ////PagingResult<Itinerary> actual;
        ////    ////using (new TransactionScope())
        ////    ////{
        ////    ////    var repository = new ItinerariesRepository();
        ////    ////    repository.Add(expected);

        ////    ////    actual = repository.Search(searchCriteria);
        ////    ////}

        ////    ////Assert.IsNotNull(actual);
        ////    ////Assert.AreNotEqual(0, actual.Items.Count);
        ////}

        ////[TestMethod]
        ////[DeploymentItem("PlanMyNight.mdf")]
        ////public void ShouldGetOnlyPublicItinerariesForActivity()
        ////{
        ////    Assert.Inconclusive();

        ////    ////var expected = new Itinerary() { Name = "Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };
        ////    ////expected.Activities.Add(new ItineraryActivity() { ActivityId = "1" });

        ////    ////var notExpected = new Itinerary() { Name = "Not Expected", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = false };
        ////    ////notExpected.Activities.Add(new ItineraryActivity() { ActivityId = "1" });

        ////    ////var searchCriteria = new AdvancedSearchQuery()
        ////    ////{
        ////    ////    ActivityTypeId = "1",
        ////    ////    City = string.Empty,
        ////    ////    Zip = string.Empty,
        ////    ////    StreetAddress = string.Empty
        ////    ////};

        ////    ////PagingResult<Itinerary> actual;
        ////    ////using (new TransactionScope())
        ////    ////{
        ////    ////    var repository = new ItinerariesRepository();
        ////    ////    repository.Add(expected);
        ////    ////    repository.Add(notExpected);

        ////    ////    actual = repository.Search(searchCriteria);
        ////    ////}

        ////    ////Assert.IsNotNull(actual);
        ////    ////Assert.AreNotEqual(0, actual.Items.Count);
        ////    ////Assert.IsTrue(actual.Items.Any(i => i.Id == expected.Id));
        ////    ////Assert.IsFalse(actual.Items.Any(i => i.Id == notExpected.Id));
        ////}

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void GetUserName()
        {
            Guid userId = Constants.TestUserId;
            var repo = new ItinerariesRepository();

            string displayName = repo.GetUserDisplayName(userId);

            Assert.AreEqual(Constants.TestUserName, displayName);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void GetUserNameReturnsNull()
        {
            var repo = new ItinerariesRepository();

            // This user Id should not exists in the database
            var userId = new Guid("{992ACB15-FA2C-43BA-9C55-54CA8DAD90E4}");

            string displayName = repo.GetUserDisplayName(userId);

            Assert.IsNull(displayName);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldReturnFalseToUserCanRateItinerary()
        {
            using (new TransactionScope())
            {
                var userId = Guid.NewGuid();
                var ctx = new PlanMyNightEntities();
                var itinerary = new Itinerary { Name = "Test", Created = DateTime.Now };
                ctx.Itineraries.AddObject(itinerary);
                ctx.SaveChanges();

                ctx.ItineraryRatings.AddObject(new ItineraryRating { ItineraryId = itinerary.Id, UserId = userId, Rating = 5, Timestamp = DateTime.Now.AddDays(-1) });
                ctx.SaveChanges();

                var repository = new ItinerariesRepository();
                var result = repository.CanUserRateItinerary(itinerary.Id, userId);

                Assert.IsFalse(result);
            }
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldReturnTrueToUserCanRateItinerary()
        {
            var repository = new ItinerariesRepository();
            var result = repository.CanUserRateItinerary(1, Guid.NewGuid());

            Assert.IsTrue(result);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void RateItineraryUpdateAverage()
        {
            using (new TransactionScope())
            {
                var ctx = new PlanMyNightEntities();
                var itinerary = new Itinerary { Name = "Test", Created = DateTime.Now };
                ctx.Itineraries.AddObject(itinerary);
                ctx.SaveChanges();

                var repository = new ItinerariesRepository();
                repository.RateItinerary(itinerary.Id, Guid.NewGuid(), 5, DateTime.Now);
                repository.RateItinerary(itinerary.Id, Guid.NewGuid(), 4, DateTime.Now);
                repository.RateItinerary(itinerary.Id, Guid.NewGuid(), 3, DateTime.Now);
                repository.RateItinerary(itinerary.Id, Guid.NewGuid(), 2, DateTime.Now);

                itinerary = repository.Retrieve(itinerary.Id);
                var totalRates = ctx.ItineraryRatings.Where(r => r.ItineraryId == itinerary.Id).Count();

                Assert.AreEqual(4, totalRates);
                Assert.AreEqual(4, itinerary.RatingCount);
                Assert.AreEqual((decimal)3.5, itinerary.Rating);
            }
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldInsertComment()
        {
            var mockItinerary = new Itinerary() { Name = "My Test Itinerary", Created = DateTime.Now, UserId = Guid.NewGuid(), IsPublic = true };

            var comment = new ItineraryComment
            {
                UserId = Constants.TestUserId,
                Body = "This is a test comment",
                Timestamp = DateTime.Now,
                IpAddress = "127.0.0.1"
            };

            using (new TransactionScope())
            {
                using (var ctx = new PlanMyNightEntities())
                {
                    ctx.Itineraries.AddObject(mockItinerary);
                    ctx.SaveChanges();
                }

                comment.ItineraryId = mockItinerary.Id;

                var repository = new ItinerariesRepository();
                repository.AddComment(comment);
            }

            Assert.AreNotEqual(0, comment.Id);
        }

        [TestMethod]
        [DeploymentItem("PlanMyNight.mdf")]
        public void ShouldRetrieveComments()
        {
            var mockItinerary = new Itinerary() { Name = "My Test Itinerary", Created = DateTime.Today, UserId = Guid.NewGuid(), IsPublic = true };

            var expected = new ItineraryComment
            {
                UserId = Constants.TestUserId,
                Body = "This is a test comment",
                Timestamp = DateTime.Today,
                IpAddress = "127.0.0.1"
            };

            ItineraryComment actual;
            using (new TransactionScope())
            {
                using (var ctx = new PlanMyNightEntities())
                {
                    ctx.Itineraries.AddObject(mockItinerary);
                    ctx.SaveChanges();
                }

                expected.ItineraryId = mockItinerary.Id;

                var repository = new ItinerariesRepository();
                repository.AddComment(expected);

                actual = repository.RetrieveComments(mockItinerary.Id).First(c => c.Id == expected.Id);
            }

            Assert.AreEqual(expected.Body, actual.Body);
            Assert.AreEqual(expected.ItineraryId, actual.ItineraryId);
            Assert.AreEqual(expected.IpAddress, actual.IpAddress);
            Assert.AreEqual(expected.Timestamp, actual.Timestamp);
            Assert.AreEqual(expected.UserId, actual.UserId);
            Assert.AreEqual(Constants.TestUserName, actual.DisplayName);  // from mock data
        }
    }
}
